import { api, APIError } from "encore.dev/api";
import { SQLDatabase } from "encore.dev/storage/sqldb";

const db = SQLDatabase.named("madrasah");

interface ImportGuruRequest {
  madrasah_id: string;
  data: GuruImportData[];
}

interface GuruImportData {
  nip?: string;
  nama_lengkap: string;
  tempat_lahir?: string;
  tanggal_lahir?: string;
  jenis_kelamin?: string;
  agama?: string;
  alamat?: string;
  no_telepon?: string;
  email?: string;
  pendidikan_terakhir?: string;
  mata_pelajaran?: string;
  status_kepegawaian?: string;
  tanggal_masuk?: string;
}

interface ImportResult {
  success: number;
  failed: number;
  errors: string[];
}

// Imports multiple guru records
export const importGuru = api<ImportGuruRequest, ImportResult>(
  { expose: true, method: "POST", path: "/guru/import" },
  async (req) => {
    const { madrasah_id, data } = req;
    
    let success = 0;
    let failed = 0;
    const errors: string[] = [];
    
    console.log("Import guru request received:", { madrasah_id, dataCount: data.length });
    
    for (let i = 0; i < data.length; i++) {
      const guru = data[i];
      try {
        console.log(`Processing guru ${i + 1}:`, guru);
        
        if (!guru.nama_lengkap || guru.nama_lengkap.trim() === "") {
          throw new Error("Nama lengkap is required");
        }
        
        // Parse tanggal_lahir if provided
        let tanggalLahir = null;
        if (guru.tanggal_lahir && guru.tanggal_lahir.trim() !== "") {
          const dateStr = guru.tanggal_lahir.trim();
          // Try to parse various date formats
          const date = new Date(dateStr);
          if (!isNaN(date.getTime())) {
            tanggalLahir = date.toISOString().split('T')[0]; // Format as YYYY-MM-DD
          }
        }
        
        // Parse tanggal_masuk if provided
        let tanggalMasuk = null;
        if (guru.tanggal_masuk && guru.tanggal_masuk.trim() !== "") {
          const dateStr = guru.tanggal_masuk.trim();
          const date = new Date(dateStr);
          if (!isNaN(date.getTime())) {
            tanggalMasuk = date.toISOString().split('T')[0];
          }
        }
        
        await db.exec`
          INSERT INTO guru (madrasah_id, nip, nama_lengkap, tempat_lahir, tanggal_lahir, jenis_kelamin, agama, alamat, 
                           no_telepon, email, pendidikan_terakhir, mata_pelajaran, status_kepegawaian, tanggal_masuk)
          VALUES (${madrasah_id}, ${(guru.nip || "").trim()}, ${guru.nama_lengkap.trim()}, ${(guru.tempat_lahir || "").trim()}, 
                  ${tanggalLahir}, ${(guru.jenis_kelamin || "").trim()}, ${(guru.agama || "").trim()}, ${(guru.alamat || "").trim()}, 
                  ${(guru.no_telepon || "").trim()}, ${(guru.email || "").trim()}, ${(guru.pendidikan_terakhir || "").trim()}, 
                  ${(guru.mata_pelajaran || "").trim()}, ${(guru.status_kepegawaian || "").trim()}, ${tanggalMasuk})
        `;
        success++;
        console.log(`Successfully imported guru ${i + 1}`);
      } catch (error) {
        failed++;
        const errorMessage = error instanceof Error ? error.message : 'Unknown error';
        errors.push(`Row ${i + 1}: ${errorMessage}`);
        console.error(`Failed to import guru ${i + 1}:`, errorMessage);
      }
    }
    
    console.log(`Import guru completed. Success: ${success}, Failed: ${failed}`);
    return { success, failed, errors };
  }
);
